# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


from google.cloud import aiplatform
from google.cloud.aiplatform.gapic.schema import predict

class VertexAITools:
    def __init__(self, project_id: str, location: str):
        self.project_id = project_id
        self.location = location
        aiplatform.init(project=self.project_id, location=self.location)

    def generate_text(self, prompt: str, model_name: str = "gemini-2.5-flash", temperature: float = 0.7, max_output_tokens: int = 1024) -> str:
        """
        Generates text using a Vertex AI generative model.

        Args:
            prompt: The input prompt for text generation.
            model_name: The name of the Vertex AI model to use (e.g., 'gemini-2.5-flash').
            temperature: Controls the randomness of the output. Lower values make output more deterministic.
            max_output_tokens: The maximum number of tokens to generate.

        Returns:
            The generated text string.
        """
        try:
            model = aiplatform.GenerativeModel(model_name)
            response = model.generate_content(
                prompt,
                generation_config={
                    "temperature": temperature,
                    "max_output_tokens": max_output_tokens,
                }
            )
            return response.text
        except Exception as e:
            print(f"Error generating text with Vertex AI: {e}")
            return ""

    def generate_content_with_safety_settings(self, prompt: str, model_name: str = "gemini-2.5-flash", temperature: float = 0.7, max_output_tokens: int = 1024) -> str:
        """
        Generates text using a Vertex AI generative model with safety settings.

        Args:
            prompt: The input prompt for text generation.
            model_name: The name of the Vertex AI model to use (e.g., 'gemini-2.5-flash').
            temperature: Controls the randomness of the output. Lower values make output more deterministic.
            max_output_tokens: The maximum number of tokens to generate.

        Returns:
            The generated text string.
        """
        safety_settings = {
            "HARM_CATEGORY_HARASSMENT": "BLOCK_NONE",
            "HARM_CATEGORY_HATE_SPEECH": "BLOCK_NONE",
            "HARM_CATEGORY_SEXUALLY_EXPLICIT": "BLOCK_NONE",
            "HARM_CATEGORY_DANGEROUS_CONTENT": "BLOCK_NONE",
        }
        try:
            model = aiplatform.GenerativeModel(model_name)
            response = model.generate_content(
                prompt,
                generation_config={
                    "temperature": temperature,
                    "max_output_tokens": max_output_tokens,
                },
                safety_settings=safety_settings
            )
            return response.text
        except Exception as e:
            print(f"Error generating text with Vertex AI (with safety settings): {e}")
            return ""

