# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


import unittest
from unittest.mock import MagicMock

from langchain_google_genai import ChatGoogleGenerativeAI
from mailing_list_engagement.agents.personalization_agent import PersonalizationAgent

class TestPersonalizationAgent(unittest.TestCase):

    def setUp(self):
        """Set up test environment."""
        self.mock_llm = MagicMock(spec=ChatGoogleGenerativeAI)
        self.personalization_agent = PersonalizationAgent(llm=self.mock_llm)

    def test_personalize_content(self):
        """Test personalization of email content."""
        original_content = "Subject: Big Sale!\n\nHi everyone,\n\nDon't miss our amazing sale!"
        target_audience = "VIP Customers"
        expected_personalized_content = "Subject: An Exclusive Offer for Our VIP Customers!\n\nDear Valued VIP Customer,\n\nAs one of our most valued customers, we're excited to offer you an exclusive preview of our biggest sale yet!"

        self.mock_llm.invoke.return_value.content = expected_personalized_content

        personalized_content = self.personalization_agent.personalize_content(original_content, target_audience)

        self.mock_llm.invoke.assert_called_once()
        self.assertEqual(personalized_content, expected_personalized_content)

    def test_personalize_content_different_audience(self):
        """Test personalization for a different audience segment."""
        original_content = "Subject: Big Sale!\n\nHi everyone,\n\nDon't miss our amazing sale!"
        target_audience = "New Subscribers"
        expected_personalized_content = "Subject: Welcome! Your Special Offer Inside!\n\nWelcome to our community! We're thrilled to have you. Here's a special offer just for you."

        self.mock_llm.invoke.return_value.content = expected_personalized_content

        personalized_content = self.personalization_agent.personalize_content(original_content, target_audience)

        self.mock_llm.invoke.assert_called_once()
        self.assertEqual(personalized_content, expected_personalized_content)

    def test_personalize_content_empty_original(self):
        """Test personalization with empty original content."""
        original_content = ""
        target_audience = "General Audience"
        expected_personalized_content = "Subject: A Message for You\n\nHello,\n\nThis is a message for you."

        self.mock_llm.invoke.return_value.content = expected_personalized_content

        personalized_content = self.personalization_agent.personalize_content(original_content, target_audience)

        self.mock_llm.invoke.assert_called_once()
        self.assertEqual(personalized_content, expected_personalized_content)

if __name__ == '__main__':
    unittest.main()
