# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


import unittest
from unittest.mock import MagicMock

from langchain_google_genai import ChatGoogleGenerativeAI
from mailing_list_engagement.agents.list_management_agent import ListManagementAgent

class TestListManagementAgent(unittest.TestCase):

    def setUp(self):
        """Set up test environment."""
        self.mock_llm = MagicMock(spec=ChatGoogleGenerativeAI)
        self.list_agent = ListManagementAgent(llm=self.mock_llm)

    def test_manage_list_segmentation(self):
        """Test list segmentation task."""
        task = "Segment the list into active and inactive subscribers based on last login date."
        context = "Subscribers: [user1, user2, user3]\nLast login: {user1: 2023-10-20, user2: 2023-01-15, user3: 2023-10-25}"
        expected_output = "Segmented list: Active: [user1, user3], Inactive: [user2]"
        self.mock_llm.invoke.return_value.content = expected_output

        result = self.list_agent.manage_list(task_description=task, context=context)

        self.mock_llm.invoke.assert_called_once()
        self.assertEqual(result, expected_output)

    def test_manage_list_hygiene(self):
        """Test list hygiene task."""
        task = "Identify and suggest removal of duplicate email addresses."
        context = "Subscribers: [userA@example.com, userB@example.com, userA@example.com]"
        expected_output = "Duplicates found: userA@example.com. Suggest removing one instance."
        self.mock_llm.invoke.return_value.content = expected_output

        result = self.list_agent.manage_list(task_description=task, context=context)

        self.mock_llm.invoke.assert_called_once()
        self.assertEqual(result, expected_output)

    def test_manage_list_no_context(self):
        """Test list management with no additional context."""
        task = "Suggest a strategy for improving email deliverability."
        expected_output = "To improve deliverability, ensure a clean list, authenticate your domain (SPF, DKIM, DMARC), and monitor sender reputation."
        self.mock_llm.invoke.return_value.content = expected_output

        result = self.list_agent.manage_list(task_description=task)

        self.mock_llm.invoke.assert_called_once()
        self.assertEqual(result, expected_output)

if __name__ == '__main__':
    unittest.main()
