# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


import unittest
from unittest.mock import MagicMock

from langchain_google_genai import ChatGoogleGenerativeAI
from mailing_list_engagement.agents.engagement_analysis_agent import EngagementAnalysisAgent
from mailing_list_engagement.tools.google_analytics_api import GoogleAnalyticsAPI

class TestEngagementAnalysisAgent(unittest.TestCase):

    def setUp(self):
        """Set up test environment."""
        self.mock_llm = MagicMock(spec=ChatGoogleGenerativeAI)
        self.mock_analytics_tool = MagicMock(spec=GoogleAnalyticsAPI)
        self.engagement_agent = EngagementAnalysisAgent(llm=self.mock_llm, analytics_tool=self.mock_analytics_tool)

    def test_analyze_engagement(self):
        """Test if the agent correctly calls the analytics tool."""
        campaign_id = "summer_sale_2023"
        mock_metrics = {
            "campaign_id": campaign_id,
            "total_sessions": 1500,
            "total_users": 1200,
            "total_conversions": 150,
            "total_events": 5000,
            "average_conversions_per_session": 0.1
        }
        self.mock_analytics_tool.get_campaign_metrics.return_value = mock_metrics

        # Call the method that uses the tool
        retrieved_metrics = self.engagement_agent.analyze_engagement(campaign_id)

        # Assert that the tool was called correctly
        self.mock_analytics_tool.get_campaign_metrics.assert_called_once_with(campaign_id=campaign_id)
        # Assert that the returned metrics are as expected
        self.assertEqual(retrieved_metrics, mock_metrics)

    def test_summarize_analysis(self):
        """Test if the agent can summarize analysis results."""
        campaign_id = "summer_sale_2023"
        metrics_data = {
            "campaign_id": campaign_id,
            "total_sessions": 1500,
            "total_users": 1200,
            "total_conversions": 150,
            "total_events": 5000,
            "average_conversions_per_session": 0.1
        }
        expected_summary = "Campaign performance was strong, with a high number of sessions and users. Conversion rate was 10%."
        self.mock_llm.invoke.return_value.content = expected_summary

        # Call the method that uses the LLM
        summary = self.engagement_agent.summarize_analysis(campaign_id, metrics_data)

        # Assert that the LLM was called
        self.mock_llm.invoke.assert_called_once()
        # Assert the summary is as expected
        self.assertEqual(summary, expected_summary)

    def test_analyze_engagement_error(self):
        """Test handling of errors from the analytics tool."""
        campaign_id = "non_existent_campaign"
        error_message = "Campaign not found."
        self.mock_analytics_tool.get_campaign_metrics.side_effect = Exception(error_message)

        retrieved_metrics = self.engagement_agent.analyze_engagement(campaign_id)

        self.mock_analytics_tool.get_campaign_metrics.assert_called_once_with(campaign_id=campaign_id)
        self.assertIn("error", retrieved_metrics)
        self.assertEqual(retrieved_metrics["error"], error_message)

if __name__ == '__main__':
    unittest.main()
