# Copyright © 2025 Intellisol LLC. All Rights Reserved.
#
# This file is part of the Intellisol Automation System.
#
# This software is a trade secret of Intellisol LLC. It is proprietary and
# confidential information. You may not disclose this software or any part of it
# to any third party, or use it in any way not expressly authorized by the
# accompanying End-User License Agreement (EULA).
#
# UNPUBLISHED. RIGHTS RESERVED.


import unittest
import os
from unittest.mock import MagicMock

from langchain_google_genai import ChatGoogleGenerativeAI
from mailing_list_engagement.agents.content_creation_agent import ContentCreationAgent

class TestContentCreationAgent(unittest.TestCase):

    def setUp(self):
        """Set up test environment."""
        # Mock the LLM to avoid actual API calls during unit tests
        self.mock_llm = MagicMock(spec=ChatGoogleGenerativeAI)
        self.content_creation_agent = ContentCreationAgent(llm=self.mock_llm)

        # Mock environment variables if needed by the agent (though not directly here)
        os.environ["GOOGLE_API_KEY"] = "dummy_api_key"

    def tearDown(self):
        """Clean up test environment."""
        # Unset mock environment variables
        if "GOOGLE_API_KEY" in os.environ:
            del os.environ["GOOGLE_API_KEY"]

    def test_create_content(self):
        """Test if the content creation agent generates reasonable output."""
        campaign_goal = "Promote summer sale with 20% off."
        target_audience = "Existing customers."

        # Configure the mock LLM's return value
        mock_response_content = {
            "content": "Subject: Summer Sale Starts Now! ☀️\n\nHi there,\n\nGet ready for our biggest sale of the year! Enjoy 20% off all items. Shop now!"
        }
        self.mock_llm.invoke.return_value.content = mock_response_content["content"]

        # Call the agent method
        generated_content = self.content_creation_agent.create_content(campaign_goal, target_audience)

        # Assert that the LLM was called with the correct prompt and inputs
        self.mock_llm.invoke.assert_called_once()
        # You can add more specific checks on the prompt arguments if needed

        # Assert that the generated content is not empty and contains expected elements
        self.assertIsNotNone(generated_content)
        self.assertIn("Subject:", generated_content)
        self.assertIn("Sale", generated_content)
        self.assertIn("20% off", generated_content)

    def test_create_content_empty_inputs(self):
        """Test content creation with empty inputs."""
        campaign_goal = ""
        target_audience = ""

        mock_response_content = {
            "content": "Subject: A Message from Us\n\nHello,\n\nWe have a message for you."
        }
        self.mock_llm.invoke.return_value.content = mock_response_content["content"]

        generated_content = self.content_creation_agent.create_content(campaign_goal, target_audience)

        self.assertIsNotNone(generated_content)
        self.assertIn("Subject:", generated_content)

if __name__ == '__main__':
    unittest.main()
